@extends('admin.layouts.home')

@section('title', 'Orders Management')

@section('styles')
<style>
    .order-card {
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        border-radius: 10px;
        overflow: hidden;
    }
    
    .order-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
    }
    
    .status-badge {
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 500;
    }
    
    .filter-card {
        border-radius: 10px;
        margin-bottom: 20px;
    }
    
    .action-btn {
        width: 32px;
        height: 32px;
        border-radius: 50%;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }
    
    .action-btn:hover {
        transform: translateY(-2px);
    }
    
    .product-thumbnail {
        width: 40px;
        height: 40px;
        object-fit: cover;
        border-radius: 6px;
    }
</style>
@endsection

@section('content')
<div class="container-fluid px-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mt-4">Orders Management</h1>
        <div>
            <a href="{{ route('admin.marketplace.dashboard') }}" class="btn btn-outline-primary me-2">
                <i class="fas fa-chart-line me-1"></i> Dashboard
            </a>
        </div>
    </div>
    
    <!-- Filter Card -->
    <div class="card filter-card mb-4">
        <div class="card-header bg-light">
            <i class="fas fa-filter me-1"></i> Filter Orders
        </div>
        <div class="card-body">
            <form action="{{ route('admin.marketplace.orders') }}" method="GET" class="row g-3">
                <div class="col-md-3">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-select" id="status" name="status">
                        <option value="">All Status</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="processing" {{ request('status') == 'processing' ? 'selected' : '' }}>Processing</option>
                        <option value="shipped" {{ request('status') == 'shipped' ? 'selected' : '' }}>Shipped</option>
                        <option value="delivered" {{ request('status') == 'delivered' ? 'selected' : '' }}>Delivered</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="date_from" class="form-label">Date From</label>
                    <input type="date" class="form-control" id="date_from" name="date_from" value="{{ request('date_from') }}">
                </div>
                
                <div class="col-md-3">
                    <label for="date_to" class="form-label">Date To</label>
                    <input type="date" class="form-control" id="date_to" name="date_to" value="{{ request('date_to') }}">
                </div>
                
                <div class="col-md-3">
                    <label for="order_by" class="form-label">Order By</label>
                    <select class="form-select" id="order_by" name="order_by">
                        <option value="created_at" {{ request('order_by') == 'created_at' ? 'selected' : '' }}>Date Created</option>
                        <option value="total_amount" {{ request('order_by') == 'total_amount' ? 'selected' : '' }}>Amount</option>
                    </select>
                </div>
                
                <div class="col-md-3">
                    <label for="order_direction" class="form-label">Direction</label>
                    <select class="form-select" id="order_direction" name="order_direction">
                        <option value="desc" {{ request('order_direction') == 'desc' ? 'selected' : '' }}>Descending</option>
                        <option value="asc" {{ request('order_direction') == 'asc' ? 'selected' : '' }}>Ascending</option>
                    </select>
                </div>
                
                <div class="col-md-6">
                    <label for="search" class="form-label">Search</label>
                    <input type="text" class="form-control" id="search" name="search" value="{{ request('search') }}" placeholder="Search by order number or customer name...">
                </div>
                
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary me-2">
                        <i class="fas fa-search me-1"></i> Apply Filters
                    </button>
                    <a href="{{ route('admin.marketplace.orders') }}" class="btn btn-outline-secondary">
                        <i class="fas fa-redo me-1"></i> Reset
                    </a>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Orders Table -->
    <div class="card mb-4">
        <div class="card-header">
            <i class="fas fa-shopping-cart me-1"></i>
            Orders List
            <span class="badge bg-primary ms-2">{{ $orders->total() }} Orders</span>
        </div>
        <div class="card-body">
            @if($orders->isEmpty())
                <div class="alert alert-info">
                    No orders found. Try adjusting your filters.
                </div>
            @else
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Order #</th>
                                <th>Customer</th>
                                <th>Products</th>
                                <th>Total</th>
                                <th>Status</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($orders as $order)
                                <tr>
                                    <td>
                                        <a href="{{ route('admin.marketplace.orders.show', $order->id) }}" class="fw-bold text-decoration-none">
                                            #{{ $order->order_number }}
                                        </a>
                                    </td>
                                    <td>
                                        @if($order->user)
                                            <div class="d-flex align-items-center">
                                                @if($order->user->avatar)
                                                    <img src="{{ asset('storage/' . $order->user->avatar) }}" alt="{{ $order->user->name }}" class="rounded-circle me-2" style="width: 32px; height: 32px; object-fit: cover;">
                                                @else
                                                    <div class="rounded-circle bg-light d-flex align-items-center justify-content-center me-2" style="width: 32px; height: 32px;">
                                                        <i class="fas fa-user text-secondary"></i>
                                                    </div>
                                                @endif
                                                <div>
                                                    {{ $order->user->name }}
                                                </div>
                                            </div>
                                        @else
                                            <span class="text-muted">Guest</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="d-flex">
                                            @foreach($order->products->take(3) as $product)
                                                <div class="position-relative me-1" data-bs-toggle="tooltip" data-bs-placement="top" title="{{ $product->name }}">
                                                    @if($product->thumbnail)
                                                        <img src="{{ asset('storage/' . $product->thumbnail) }}" alt="{{ $product->name }}" class="product-thumbnail">
                                                    @else
                                                        <div class="product-thumbnail bg-light d-flex align-items-center justify-content-center">
                                                            <i class="fas fa-box text-secondary"></i>
                                                        </div>
                                                    @endif
                                                </div>
                                            @endforeach
                                            
                                            @if($order->products->count() > 3)
                                                <div class="product-thumbnail bg-light d-flex align-items-center justify-content-center">
                                                    <span class="small">+{{ $order->products->count() - 3 }}</span>
                                                </div>
                                            @endif
                                        </div>
                                    </td>
                                    <td class="fw-bold">${{ number_format($order->total_amount, 2) }}</td>
                                    <td>
                                        @php
                                            $statusClass = [
                                                'pending' => 'bg-warning text-dark',
                                                'processing' => 'bg-info text-white',
                                                'shipped' => 'bg-primary',
                                                'delivered' => 'bg-success',
                                                'completed' => 'bg-success',
                                                'cancelled' => 'bg-danger',
                                            ][$order->status] ?? 'bg-secondary';
                                        @endphp
                                        <span class="status-badge {{ $statusClass }}">
                                            {{ ucfirst($order->status) }}
                                        </span>
                                    </td>
                                    <td>{{ $order->created_at->format('M d, Y H:i') }}</td>
                                    <td>
                                        <div class="d-flex">
                                            <a href="{{ route('admin.marketplace.orders.show', $order->id) }}" class="btn btn-sm btn-info action-btn me-1" title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            
                                            <button type="button" class="btn btn-sm btn-warning action-btn" title="Update Status" data-bs-toggle="modal" data-bs-target="#statusModal{{ $order->id }}">
                                                <i class="fas fa-sync-alt"></i>
                                            </button>
                                            
                                            <!-- Status Modal -->
                                            <div class="modal fade" id="statusModal{{ $order->id }}" tabindex="-1" aria-labelledby="statusModalLabel{{ $order->id }}" aria-hidden="true">
                                                <div class="modal-dialog">
                                                    <div class="modal-content">
                                                        <div class="modal-header">
                                                            <h5 class="modal-title" id="statusModalLabel{{ $order->id }}">Update Order Status</h5>
                                                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                        </div>
                                                        <form action="{{ route('admin.marketplace.orders.update-status', $order->id) }}" method="POST">
                                                            @csrf
                                                            <div class="modal-body">
                                                                <p>Update status for Order <strong>#{{ $order->order_number }}</strong>:</p>
                                                                <div class="form-check mb-2">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusPending{{ $order->id }}" value="pending" {{ $order->status == 'pending' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusPending{{ $order->id }}">
                                                                        Pending
                                                                    </label>
                                                                </div>
                                                                <div class="form-check mb-2">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusProcessing{{ $order->id }}" value="processing" {{ $order->status == 'processing' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusProcessing{{ $order->id }}">
                                                                        Processing
                                                                    </label>
                                                                </div>
                                                                <div class="form-check mb-2">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusShipped{{ $order->id }}" value="shipped" {{ $order->status == 'shipped' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusShipped{{ $order->id }}">
                                                                        Shipped
                                                                    </label>
                                                                </div>
                                                                <div class="form-check mb-2">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusDelivered{{ $order->id }}" value="delivered" {{ $order->status == 'delivered' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusDelivered{{ $order->id }}">
                                                                        Delivered
                                                                    </label>
                                                                </div>
                                                                <div class="form-check mb-2">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusCompleted{{ $order->id }}" value="completed" {{ $order->status == 'completed' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusCompleted{{ $order->id }}">
                                                                        Completed
                                                                    </label>
                                                                </div>
                                                                <div class="form-check">
                                                                    <input class="form-check-input" type="radio" name="status" id="statusCancelled{{ $order->id }}" value="cancelled" {{ $order->status == 'cancelled' ? 'checked' : '' }}>
                                                                    <label class="form-check-label" for="statusCancelled{{ $order->id }}">
                                                                        Cancelled
                                                                    </label>
                                                                </div>
                                                            </div>
                                                            <div class="modal-footer">
                                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                                <button type="submit" class="btn btn-primary">Update Status</button>
                                                            </div>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-4">
                    {{ $orders->appends(request()->query())->links('pagination::bootstrap-5') }}
                </div>
            @endif
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize tooltips
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl)
        });
    });
</script>
@endsection 