<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Messenger\Conversation;
use App\Models\Messenger\ConversationParticipant;
use App\Models\Messenger\Message;
use App\Models\Messenger\MessageReceipt;
use Illuminate\Support\Facades\DB;

class MessengerDemoSeeder extends Seeder
{
    public function run(): void
    {
        // Get or create demo users
        $user1 = User::firstOrCreate(
            ['email' => 'demo1@example.com'],
            [
                'username' => 'demo_user_1',
                'first_name' => 'Demo',
                'last_name' => 'User 1',
                'name' => 'Demo User 1',
                'password' => bcrypt('password'),
                'email_verified_at' => now(),
            ]
        );

        $user2 = User::firstOrCreate(
            ['email' => 'demo2@example.com'],
            [
                'username' => 'demo_user_2',
                'first_name' => 'Demo',
                'last_name' => 'User 2',
                'name' => 'Demo User 2',
                'password' => bcrypt('password'),
                'email_verified_at' => now(),
            ]
        );

        $user3 = User::firstOrCreate(
            ['email' => 'demo3@example.com'],
            [
                'username' => 'demo_user_3',
                'first_name' => 'Demo',
                'last_name' => 'User 3',
                'name' => 'Demo User 3',
                'password' => bcrypt('password'),
                'email_verified_at' => now(),
            ]
        );

        // Create a DM conversation between user1 and user2
        $dmConversation = Conversation::create([
            'type' => 'dm',
            'title' => null,
            'created_by' => $user1->id,
            'last_activity_at' => now(),
        ]);

        // Add participants
        ConversationParticipant::create([
            'conversation_id' => $dmConversation->id,
            'user_id' => $user1->id,
            'role' => 'owner',
            'joined_at' => now(),
        ]);

        ConversationParticipant::create([
            'conversation_id' => $dmConversation->id,
            'user_id' => $user2->id,
            'role' => 'member',
            'joined_at' => now(),
        ]);

        // Create some messages
        $message1 = Message::create([
            'conversation_id' => $dmConversation->id,
            'user_id' => $user2->id,
            'type' => 'text',
            'content' => 'Hey there! How are you doing?',
            'client_generated_id' => 'msg_1',
        ]);

        $message2 = Message::create([
            'conversation_id' => $dmConversation->id,
            'user_id' => $user1->id,
            'type' => 'text',
            'content' => 'I\'m doing great! Thanks for asking. How about you?',
            'client_generated_id' => 'msg_2',
        ]);

        $message3 = Message::create([
            'conversation_id' => $dmConversation->id,
            'user_id' => $user2->id,
            'type' => 'text',
            'content' => 'Pretty good! Any plans for the weekend?',
            'client_generated_id' => 'msg_3',
        ]);

        // Update conversation with last message
        $dmConversation->update([
            'last_message_id' => $message3->id,
            'last_activity_at' => $message3->created_at,
        ]);

        // Create message receipts
        MessageReceipt::create([
            'message_id' => $message1->id,
            'user_id' => $user1->id,
            'type' => 'delivered',
            'delivered_at' => $message1->created_at,
        ]);

        MessageReceipt::create([
            'message_id' => $message1->id,
            'user_id' => $user1->id,
            'type' => 'seen',
            'delivered_at' => $message1->created_at,
            'seen_at' => $message1->created_at->addMinutes(1),
        ]);

        MessageReceipt::create([
            'message_id' => $message2->id,
            'user_id' => $user2->id,
            'type' => 'delivered',
            'delivered_at' => $message2->created_at,
        ]);

        MessageReceipt::create([
            'message_id' => $message2->id,
            'user_id' => $user2->id,
            'type' => 'seen',
            'delivered_at' => $message2->created_at,
            'seen_at' => $message2->created_at->addMinutes(1),
        ]);

        // Create a group conversation
        $groupConversation = Conversation::create([
            'type' => 'group',
            'title' => 'Project Team',
            'description' => 'A group for our project team discussions',
            'created_by' => $user1->id,
            'last_activity_at' => now(),
        ]);

        // Add group participants
        ConversationParticipant::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user1->id,
            'role' => 'owner',
            'joined_at' => now(),
        ]);

        ConversationParticipant::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user2->id,
            'role' => 'admin',
            'joined_at' => now(),
        ]);

        ConversationParticipant::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user3->id,
            'role' => 'member',
            'joined_at' => now(),
        ]);

        // Create group messages
        $groupMessage1 = Message::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user1->id,
            'type' => 'text',
            'content' => 'Welcome everyone to our project team!',
            'client_generated_id' => 'group_msg_1',
        ]);

        $groupMessage2 = Message::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user2->id,
            'type' => 'text',
            'content' => 'Thanks! Looking forward to working together.',
            'client_generated_id' => 'group_msg_2',
        ]);

        $groupMessage3 = Message::create([
            'conversation_id' => $groupConversation->id,
            'user_id' => $user3->id,
            'type' => 'text',
            'content' => 'Same here! When is our first meeting?',
            'client_generated_id' => 'group_msg_3',
        ]);

        // Update group conversation with last message
        $groupConversation->update([
            'last_message_id' => $groupMessage3->id,
            'last_activity_at' => $groupMessage3->created_at,
        ]);

        // Create receipts for group messages
        foreach ([$groupMessage1, $groupMessage2, $groupMessage3] as $message) {
            foreach ([$user1, $user2, $user3] as $user) {
                if ($user->id !== $message->user_id) {
                    MessageReceipt::create([
                        'message_id' => $message->id,
                        'user_id' => $user->id,
                        'type' => 'delivered',
                        'delivered_at' => $message->created_at,
                    ]);
                }
            }
        }

        $this->command->info('Messenger demo data seeded successfully!');
        $this->command->info("Created users: {$user1->username}, {$user2->username}, {$user3->username}");
        $this->command->info("Created conversations: 1 DM, 1 Group");
        $this->command->info("Created messages: 3 in DM, 3 in Group");
    }
}
