<?php

use App\Models\Setting;

$settings = [];

if (class_exists(Setting::class)) {
    try {
        $settings = Setting::getSettings();
    } catch (\Throwable $th) {
        $settings = [];
    }
}

$get = static function (string $key, ?string $envKey = null, mixed $default = null) use ($settings) {
    if (array_key_exists($key, $settings) && $settings[$key] !== null && $settings[$key] !== '') {
        return $settings[$key];
    }

    if ($envKey !== null) {
        $envValue = env($envKey);

        if ($envValue !== null && $envValue !== '') {
            return $envValue;
        }
    }

    return $default;
};

$getBool = static function (string $key, ?string $envKey = null, bool $default = false) use ($settings) {
    if (array_key_exists($key, $settings)) {
        $value = $settings[$key];

        if (is_bool($value)) {
            return $value;
        }

        if (is_numeric($value)) {
            return (bool) ((int) $value);
        }

        $normalized = strtolower((string) $value);

        return in_array($normalized, ['1', 'true', 'yes', 'on'], true);
    }

    if ($envKey !== null) {
        $envValue = env($envKey);

        if ($envValue !== null) {
            return filter_var($envValue, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? $default;
        }
    }

    return $default;
};

$resolveBaseUrl = static function (): string {
    try {
        if (function_exists('request')) {
            $request = request();

            if ($request instanceof \Illuminate\Http\Request) {
                $host = $request->getSchemeAndHttpHost();

                if (!empty($host)) {
                    return rtrim($host, '/');
                }
            }
        }
    } catch (\Throwable $th) {
        // Ignore and use fallbacks
    }

    try {
        if (app()->bound('url')) {
            $url = app('url')->to('/');

            if (!empty($url)) {
                return rtrim($url, '/');
            }
        }
    } catch (\Throwable $th) {
        // Ignore and use fallbacks
    }

    return rtrim(env('APP_URL', 'http://localhost'), '/');
};

$baseCallbackUrl = $resolveBaseUrl();

$getRedirect = static function (string $provider) use ($baseCallbackUrl) {
    return "{$baseCallbackUrl}/login/{$provider}/callback";
};

return [

    /*
    |--------------------------------------------------------------------------
    | Third Party Services
    |--------------------------------------------------------------------------
    |
    | This file is for storing the credentials for third party services such
    | as Mailgun, Postmark, AWS and more. This file provides the de facto
    | location for this type of information, allowing packages to have
    | a conventional file to locate the various service credentials.
    |
    */

    'postmark' => [
        'token' => env('POSTMARK_TOKEN'),
    ],

    'ses' => [
        'key' => env('AWS_ACCESS_KEY_ID'),
        'secret' => env('AWS_SECRET_ACCESS_KEY'),
        'region' => env('AWS_DEFAULT_REGION', 'us-east-1'),
    ],

    'slack' => [
        'notifications' => [
            'bot_user_oauth_token' => env('SLACK_BOT_USER_OAUTH_TOKEN'),
            'channel' => env('SLACK_BOT_USER_DEFAULT_CHANNEL'),
        ],
    ],
    'facebook' => [
        'enabled' => $getBool('chck-facebookLogin', 'FACEBOOK_LOGIN_ENABLED'),
        'client_id' => $get('facebookAppId', 'FACEBOOK_CLIENT_ID'),
        'client_secret' => $get('facebookAppKey', 'FACEBOOK_CLIENT_SECRET'),
        'redirect' => $getRedirect('facebook'),
    ],

    'google' => [
        'enabled' => $getBool('chck-googleLogin', 'GOOGLE_LOGIN_ENABLED'),
        'client_id' => $get('googleAppId', 'GOOGLE_CLIENT_ID'),
        'client_secret' => $get('googleAppKey', 'GOOGLE_CLIENT_SECRET'),
        'redirect' => $getRedirect('google'),
    ],

    'twitter' => [
        'enabled' => $getBool('chck-twitterLogin', 'TWITTER_LOGIN_ENABLED'),
        'client_id' => $get('twitterAppId', 'TWITTER_CLIENT_ID'),
        'client_secret' => $get('twitterAppKey', 'TWITTER_CLIENT_SECRET'),
        'redirect' => $getRedirect('twitter'),
    ],

    'linkedin' => [
        'enabled' => $getBool('chck-linkedinLogin', 'LINKEDIN_LOGIN_ENABLED'),
        'client_id' => $get('linkedinAppId', 'LINKEDIN_CLIENT_ID'),
        'client_secret' => $get('linkedinAppKey', 'LINKEDIN_CLIENT_SECRET'),
        'redirect' => $getRedirect('linkedin'),
    ],

    'vkontakte' => [
        'enabled' => $getBool('chck-VkontakteLogin', 'VKONTAKTE_LOGIN_ENABLED'),
        'client_id' => $get('VkontakteAppId', 'VKONTAKTE_CLIENT_ID'),
        'client_secret' => $get('VkontakteAppKey', 'VKONTAKTE_CLIENT_SECRET'),
        'redirect' => $getRedirect('vkontakte'),
    ],

    'instagram' => [
        'enabled' => $getBool('chck-instagramLogin', 'INSTAGRAM_LOGIN_ENABLED'),
        'client_id' => $get('instagramAppId', 'INSTAGRAM_CLIENT_ID'),
        'client_secret' => $get('instagramAppkey', 'INSTAGRAM_CLIENT_SECRET'),
        'redirect' => $getRedirect('instagram'),
    ],

    'qq' => [
        'enabled' => $getBool('chck-qqLogin', 'QQ_LOGIN_ENABLED'),
        'client_id' => $get('qqAppId', 'QQ_CLIENT_ID'),
        'client_secret' => $get('qqAppkey', 'QQ_CLIENT_SECRET'),
        'redirect' => $getRedirect('qq'),
    ],

    'wechat' => [
        'enabled' => $getBool('chck-WeChatLogin', 'WECHAT_LOGIN_ENABLED'),
        'client_id' => $get('WeChatAppId', 'WECHAT_CLIENT_ID'),
        'client_secret' => $get('WeChatAppkey', 'WECHAT_CLIENT_SECRET'),
        'redirect' => $getRedirect('wechat'),
    ],

    'discord' => [
        'enabled' => $getBool('chck-DiscordLogin', 'DISCORD_LOGIN_ENABLED'),
        'client_id' => $get('DiscordAppId', 'DISCORD_CLIENT_ID'),
        'client_secret' => $get('DiscordAppkey', 'DISCORD_CLIENT_SECRET'),
        'redirect' => $getRedirect('discord'),
    ],

    'mailru' => [
        'enabled' => $getBool('chck-MailruLogin', 'MAILRU_LOGIN_ENABLED'),
        'client_id' => $get('MailruAppId', 'MAILRU_CLIENT_ID'),
        'client_secret' => $get('MailruAppkey', 'MAILRU_CLIENT_SECRET'),
        'redirect' => $getRedirect('mailru'),
    ],

    'odnoklassniki' => [
        'enabled' => $getBool('chck-OkLogin', 'OK_LOGIN_ENABLED'),
        'client_id' => $get('OkAppId', 'OK_CLIENT_ID'),
        'client_secret' => $get('OkAppSecretkey', 'OK_CLIENT_SECRET'),
        'client_public' => $get('OkAppPublickey', 'OK_CLIENT_PUBLIC'),
        'redirect' => $getRedirect('odnoklassniki'),
    ],

    'tiktok' => [
        'enabled' => $getBool('chck-tiktok_login', 'TIKTOK_LOGIN_ENABLED'),
        'client_key' => $get('tiktok_client_key', 'TIKTOK_CLIENT_KEY'),
        'client_secret' => $get('tiktok_client_secret', 'TIKTOK_CLIENT_SECRET'),
        'redirect' => $getRedirect('tiktok'),
    ],

    'wordpress' => [
        'enabled' => $getBool('chck-wordpress_login', 'WORDPRESS_LOGIN_ENABLED'),
        'client_id' => $get('WordPressAppId', 'WORDPRESS_CLIENT_ID'),
        'client_secret' => $get('WordPressAppkey', 'WORDPRESS_CLIENT_SECRET'),
        'redirect' => $getRedirect('wordpress'),
    ],

    'github' => [
        'client_id' => $get('githubClientId', 'GITHUB_CLIENT_ID'),
        'client_secret' => $get('githubClientSecret', 'GITHUB_CLIENT_SECRET'),
        'redirect' => $getRedirect('github'),
    ],

    'paypal' => [
        'mode' => env('PAYPAL_MODE', 'sandbox'),
        'client_id' => env('PAYPAL_CLIENT_ID'),
        'client_secret' => env('PAYPAL_CLIENT_SECRET'),
        'currency' => env('PAYPAL_CURRENCY', 'USD'),
    ],
];
