<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\Post;
use App\Models\User;
use App\Models\Group;
use App\Models\Page;

class NotificationService
{
    /**
     * Send friend request notification
     */
    public static function sendFriendRequest(User $sender, User $receiver)
    {
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'friend_request',
            'text' => "{$sender->name} sent you a friend request.",
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send friend accept notification
     */
    public static function sendFriendAccept(User $sender, User $receiver)
    {
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'friend_accept',
            'text' => "{$sender->name} accepted your friend request.",
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send post like notification
     */
    public static function sendPostLike(User $sender, User $receiver, Post $post)
    {
        // Don't notify yourself
        if ($sender->id === $receiver->id) {
            return null;
        }
        
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'post_like',
            'text' => "{$sender->name} liked your post.",
            'post_id' => $post->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send post comment notification
     */
    public static function sendPostComment(User $sender, User $receiver, Post $post)
    {
        // Don't notify yourself
        if ($sender->id === $receiver->id) {
            return null;
        }
        
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'post_comment',
            'text' => "{$sender->name} commented on your post.",
            'post_id' => $post->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send comment reply notification
     */
    public static function sendCommentReply(User $sender, User $receiver, Post $post)
    {
        // Don't notify yourself
        if ($sender->id === $receiver->id) {
            return null;
        }
        
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'comment_reply',
            'text' => "{$sender->name} replied to your comment.",
            'post_id' => $post->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send post mention notification
     */
    public static function sendPostMention(User $sender, User $receiver, Post $post)
    {
        // Don't notify yourself
        if ($sender->id === $receiver->id) {
            return null;
        }
        
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'post_mention',
            'text' => "{$sender->name} mentioned you in a post.",
            'post_id' => $post->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send group invite notification
     */
    public static function sendGroupInvite(User $sender, User $receiver, Group $group)
    {
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'group_invite',
            'text' => "{$sender->name} invited you to join {$group->name} group.",
            'group_id' => $group->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send group join request notification
     */
    public static function sendGroupJoinRequest(User $sender, User $receiver, Group $group)
    {
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => 'group_join_request',
            'text' => "{$sender->name} requested to join {$group->name} group.",
            'group_id' => $group->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send page like notification
     */
    public static function sendPageLike(User $sender, User $pageOwner, Page $page)
    {
        // Don't notify yourself
        if ($sender->id === $pageOwner->id) {
            return null;
        }
        
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $pageOwner->id,
            'type' => 'page_like',
            'text' => "{$sender->name} liked your page {$page->name}.",
            'page_id' => $page->id,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
    
    /**
     * Send custom notification
     */
    public static function sendCustom(User $sender, User $receiver, $type, $text, $postId = null, $pageId = null, $groupId = null)
    {
        return Notification::create([
            'from_user_id' => $sender->id,
            'to_user_id' => $receiver->id,
            'type' => $type,
            'text' => $text,
            'post_id' => $postId,
            'page_id' => $pageId,
            'group_id' => $groupId,
            'seen' => false,
            'is_reacted' => false,
        ]);
    }
}