<?php
namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\RewardSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class RewardSettingController extends Controller
{
    /**
     * Display a listing of the reward settings.
     */
    public function index()
    {
        $rewardSettings = RewardSetting::latest()->paginate(15);
        return view('admin.reward_settings.index', compact('rewardSettings'));
    }

    /**
     * Show the form for creating a new reward setting.
     */
    public function create()
    {
        return view('admin.reward_settings.create');
    }

    /**
     * Store a newly created reward setting in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action_type' => 'required|string|unique:reward_settings',
            'reward_amount' => 'required|numeric|min:0',
            'daily_limit' => 'required|integer|min:0',
            'monthly_limit' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
            'conditions' => 'nullable|json'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        RewardSetting::create([
            'action_type' => $request->action_type,
            'reward_amount' => $request->reward_amount,
            'daily_limit' => $request->daily_limit,
            'monthly_limit' => $request->monthly_limit,
            'is_active' => $request->is_active ?? false,
            'description' => $request->description,
            'conditions' => $request->conditions
        ]);

        return redirect()->route('admin.reward-settings.index')
            ->with('success', 'Reward setting created successfully.');
    }

    /**
     * Show the form for editing the specified reward setting.
     */
    public function edit(RewardSetting $rewardSetting)
    {
        return view('admin.reward_settings.edit', compact('rewardSetting'));
    }

    /**
     * Update the specified reward setting in storage.
     */
    public function update(Request $request, RewardSetting $rewardSetting)
    {
        $validator = Validator::make($request->all(), [
            'action_type' => 'required|string|unique:reward_settings,action_type,' . $rewardSetting->id,
            'reward_amount' => 'required|numeric|min:0',
            'daily_limit' => 'required|integer|min:0',
            'monthly_limit' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
            'conditions' => 'nullable|json'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $rewardSetting->update([
            'action_type' => $request->action_type,
            'reward_amount' => $request->reward_amount,
            'daily_limit' => $request->daily_limit,
            'monthly_limit' => $request->monthly_limit,
            'is_active' => $request->is_active ?? false,
            'description' => $request->description,
            'conditions' => $request->conditions
        ]);

        return redirect()->route('admin.reward-settings.index')
            ->with('success', 'Reward setting updated successfully.');
    }

    /**
     * Remove the specified reward setting from storage.
     */
    public function destroy(RewardSetting $rewardSetting)
    {
        $rewardSetting->delete();

        return redirect()->route('admin.reward-settings.index')
            ->with('success', 'Reward setting deleted successfully.');
    }
} 