<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\JobCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class JobCategoryController extends Controller
{
    /**
     * Display a listing of all job categories.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $categories = JobCategory::withCount('jobs')->latest()->get();
        return view('admin.pages.job_categories.index', compact('categories'));
    }

    /**
     * Show the form for creating a new job category.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.pages.job_categories.create');
    }

    /**
     * Store a newly created job category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|boolean',
        ]);

        JobCategory::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'status' => $request->status,
        ]);

        return redirect()->route('admin.job-categories.index')
            ->with('success', 'Job category created successfully.');
    }

    /**
     * Show the form for editing the specified job category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $jobCategory = JobCategory::findOrFail($id);
        return view('admin.pages.job_categories.edit', compact('jobCategory'));
    }

    /**
     * Update the specified job category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $jobCategory = JobCategory::findOrFail($id);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|boolean',
        ]);

        $jobCategory->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'status' => $request->status,
        ]);

        return redirect()->route('admin.job-categories.index')
            ->with('success', 'Job category updated successfully.');
    }

    /**
     * Toggle the status of the job category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function toggleStatus($id)
    {
        $jobCategory = JobCategory::findOrFail($id);
        $jobCategory->status = !$jobCategory->status;
        $jobCategory->save();
        
        $statusText = $jobCategory->status ? 'activated' : 'deactivated';
        
        return redirect()->back()
            ->with('success', "Job category {$statusText} successfully.");
    }

    /**
     * Remove the specified job category from storage.
     *
     * @param  \App\Models\JobCategory  $jobCategory
     * @return \Illuminate\Http\Response
     */
    public function destroy(JobCategory $jobCategory)
    {
        // Check if category has any associated jobs
        if ($jobCategory->jobs()->count() > 0) {
            return redirect()->route('admin.job-categories.index')
                ->with('error', 'Cannot delete this category as it has jobs associated with it.');
        }
        
        $jobCategory->delete();

        return redirect()->route('admin.job-categories.index')
            ->with('success', 'Job category deleted successfully.');
    }
} 