<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\HelpCenterTopic;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class HelpCenterTopicController extends Controller
{
    public function index()
    {
        $topics = HelpCenterTopic::query()
            ->orderBy('display_order')
            ->orderBy('title')
            ->paginate(15);

        return view('admin.pages.help_center.topics.index', compact('topics'));
    }

    public function create()
    {
        return view('admin.pages.help_center.topics.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:help_center_topics,slug',
            'icon' => 'nullable|string|max:120',
            'description' => 'nullable|string',
            'display_order' => 'nullable|integer|min:0',
            'is_active' => 'sometimes|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $validated['slug'] = $validated['slug'] ?? Str::slug($validated['title']);
        $validated['is_active'] = $request->boolean('is_active');
        $validated['display_order'] = $validated['display_order'] ?? 0;

        HelpCenterTopic::create($validated);

        return redirect()
            ->route('admin.help-center.topics.index')
            ->with('success', 'Help center topic created successfully.');
    }

    public function edit(HelpCenterTopic $topic)
    {
        return view('admin.pages.help_center.topics.edit', compact('topic'));
    }

    public function update(Request $request, HelpCenterTopic $topic)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:help_center_topics,slug,' . $topic->id,
            'icon' => 'nullable|string|max:120',
            'description' => 'nullable|string',
            'display_order' => 'nullable|integer|min:0',
            'is_active' => 'sometimes|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $validated['slug'] = $validated['slug'] ?? Str::slug($validated['title']);
        $validated['is_active'] = $request->boolean('is_active');
        $validated['display_order'] = $validated['display_order'] ?? 0;

        $topic->update($validated);

        return redirect()
            ->route('admin.help-center.topics.index')
            ->with('success', 'Help center topic updated successfully.');
    }

    public function destroy(HelpCenterTopic $topic)
    {
        if ($topic->articles()->exists()) {
            return redirect()
                ->route('admin.help-center.topics.index')
                ->with('error', 'Please remove articles under this topic before deleting.');
        }

        $topic->delete();

        return redirect()
            ->route('admin.help-center.topics.index')
            ->with('success', 'Help center topic deleted successfully.');
    }
}
