<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\CustomPage;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CustomPageController extends Controller
{
    /**
     * Display a listing of custom pages.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $pages = CustomPage::latest()->paginate(11);
        return view('admin.pages.custom_pages.index', compact('pages'));
    }

    /**
     * Show the form for creating a new custom page.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('admin.pages.custom_pages.create');
    }

    /**
     * Store a newly created custom page in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'slug' => 'nullable|string|max:255|unique:custom_pages,slug',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'active' => 'boolean',
            'show_in_footer' => 'boolean',
            'show_in_header' => 'boolean',
            'order' => 'nullable|integer',
        ]);

        // Create slug from title if slug is not provided
        $slug = $request->slug ? Str::slug($request->slug) : Str::slug($request->title);

        CustomPage::create([
            'title' => $request->title,
            'slug' => $slug,
            'content' => $request->content,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'active' => $request->active ?? false,
            'show_in_footer' => $request->show_in_footer ?? false,
            'show_in_header' => $request->show_in_header ?? false,
            'order' => $request->order ?? 0,
        ]);

        return redirect()->route('admin.custom-pages.index')
            ->with('success', 'Page created successfully.');
    }

    /**
     * Display the specified custom page.
     *
     * @param  \App\Models\CustomPage  $customPage
     * @return \Illuminate\View\View
     */
    public function show(CustomPage $customPage)
    {
        return view('admin.pages.custom_pages.show', compact('customPage'));
    }

    /**
     * Show the form for editing the specified custom page.
     *
     * @param  \App\Models\CustomPage  $customPage
     * @return \Illuminate\View\View
     */
    public function edit(CustomPage $customPage)
    {
        return view('admin.pages.custom_pages.edit', compact('customPage'));
    }

    /**
     * Update the specified custom page in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\CustomPage  $customPage
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, CustomPage $customPage)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'slug' => 'nullable|string|max:255|unique:custom_pages,slug,' . $customPage->id,
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'active' => 'boolean',
            'show_in_footer' => 'boolean',
            'show_in_header' => 'boolean',
            'order' => 'nullable|integer',
        ]);

        // Create slug from title if slug is not provided
        $slug = $request->slug ? Str::slug($request->slug) : Str::slug($request->title);

        $customPage->update([
            'title' => $request->title,
            'slug' => $slug,
            'content' => $request->content,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'active' => $request->active ?? false,
            'show_in_footer' => $request->show_in_footer ?? false,
            'show_in_header' => $request->show_in_header ?? false,
            'order' => $request->order ?? 0,
        ]);

        return redirect()->route('admin.custom-pages.index')
            ->with('success', 'Page updated successfully.');
    }

    /**
     * Remove the specified custom page from storage.
     *
     * @param  \App\Models\CustomPage  $customPage
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(CustomPage $customPage)
    {
        $customPage->delete();

        return redirect()->route('admin.custom-pages.index')
            ->with('success', 'Page deleted successfully.');
    }

    /**
     * Display a listing of trashed custom pages.
     *
     * @return \Illuminate\View\View
     */
    public function trashed()
    {
        $trashedPages = CustomPage::onlyTrashed()->paginate(10);
        return view('admin.pages.custom_pages.trashed', compact('trashedPages'));
    }

    /**
     * Restore the specified custom page from trash.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function restore($id)
    {
        $customPage = CustomPage::onlyTrashed()->findOrFail($id);
        $customPage->restore();

        return redirect()->route('admin.custom-pages.index')
            ->with('success', 'Page restored successfully.');
    }

    /**
     * Permanently delete the specified custom page from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function forceDelete($id)
    {
        $customPage = CustomPage::onlyTrashed()->findOrFail($id);
        $customPage->forceDelete();

        return redirect()->route('admin.custom-pages.trashed')
            ->with('success', 'Page permanently deleted.');
    }
}
