<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Courses\CourseCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CourseCategoryController extends Controller
{
    /**
     * Display a listing of all course categories.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $categories = CourseCategory::withCount('courses')->latest()->get();
        return view('admin.pages.course_categories.index', compact('categories'));
    }

    /**
     * Show the form for creating a new course category.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.pages.course_categories.create');
    }

    /**
     * Store a newly created course category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:course_categories,name',
            'description' => 'nullable|string',
            'icon' => 'nullable|string|max:50',
        ]);

        CourseCategory::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.course-categories.index')
            ->with('success', 'Course category created successfully.');
    }

    /**
     * Show the form for editing the specified course category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $category = CourseCategory::findOrFail($id);
        return view('admin.pages.course_categories.edit', compact('category'));
    }

    /**
     * Update the specified course category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $category = CourseCategory::findOrFail($id);
        
        $request->validate([
            'name' => 'required|string|max:255|unique:course_categories,name,' . $id,
            'description' => 'nullable|string',
            'icon' => 'nullable|string|max:50',
        ]);

        $category->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.course-categories.index')
            ->with('success', 'Course category updated successfully.');
    }

    /**
     * Toggle the status of the course category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function toggleStatus($id)
    {
        $category = CourseCategory::findOrFail($id);
        $category->is_active = !$category->is_active;
        $category->save();
        
        $statusText = $category->is_active ? 'activated' : 'deactivated';
        
        return redirect()->back()
            ->with('success', "Course category {$statusText} successfully.");
    }

    /**
     * Remove the specified course category from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $category = CourseCategory::findOrFail($id);
        
        // Check if category has any associated courses
        if ($category->courses()->count() > 0) {
            return redirect()->route('admin.course-categories.index')
                ->with('error', 'Cannot delete this category as it has courses associated with it.');
        }
        
        $category->delete();

        return redirect()->route('admin.course-categories.index')
            ->with('success', 'Course category deleted successfully.');
    }
}

